/* ==== VALIDAZIONI SEZIONE DATI BANCARI ==== */
"use strict";

// ✅ Regex IBAN italiano standard (27 caratteri totali, IT + 2 cifre di controllo + 1 lettera + 10 cifre + 12 alfanumerici)
var reg_iban = /^IT\d{2}[A-Z]\d{10}[0-9A-Z]{12}$/;

/* -------------------------------------------------
   Helper
--------------------------------------------------*/
// Normalizza IBAN: rimuove spazi e forza maiuscolo
function normalizeIban(s) {
  return (s || "").toString().replace(/\s+/g, "").toUpperCase();
}

// Verifica Modulo 97 (ISO 13616)
// 1) sposta le prime 4 cifre in fondo
// 2) converte lettere in numeri (A=10 ... Z=35)
// 3) calcola il resto in modo incrementale (evita overflow)
function ibanModulo97Ok(iban) {
  var clean = normalizeIban(iban);
  if (clean.length < 4) return false;

  var rearranged = clean.slice(4) + clean.slice(0, 4);

  var remainder = 0;
  for (var i = 0; i < rearranged.length; i++) {
    var ch = rearranged.charAt(i);
    var code = rearranged.charCodeAt(i);

    var chunk;
    if (code >= 65 && code <= 90) {          // A-Z
      chunk = String(code - 55);             // 'A'->10 ... 'Z'->35
    } else if (code >= 48 && code <= 57) {   // 0-9
      chunk = ch;
    } else {
      return false; // carattere non valido
    }

    // Integra il "chunk" cifra per cifra nel modulo
    for (var j = 0; j < chunk.length; j++) {
      remainder = (remainder * 10 + (chunk.charCodeAt(j) - 48)) % 97;
    }
  }
  return remainder === 1;
}

/* -------------------------------------------------
   VALIDAZIONI LIVE
--------------------------------------------------*/

// IBAN
$(document).on("input", "#iban", function () {
  var iban = normalizeIban($(this).val());
  $(this).val(iban); // forza maiuscolo e toglie spazi

  if (iban.length === 0) {
    $("#iban").addClass("is-invalid");
    $("#errore_iban").addClass("invalid-feedback").text("IBAN obbligatorio.");
    return;
  }

  // IBAN IT deve essere di 27 caratteri
  if (iban.startsWith("IT") && iban.length !== 27) {
    $("#iban").addClass("is-invalid");
    $("#errore_iban").addClass("invalid-feedback").text("L’IBAN italiano deve contenere 27 caratteri.");
    return;
  }

  // Struttura
  if (!reg_iban.test(iban)) {
    $("#iban").addClass("is-invalid");
    $("#errore_iban").addClass("invalid-feedback").text("IBAN non valido. Deve iniziare con 'IT' e rispettare la struttura standard.");
    return;
  }

  // ✅ Modulo 97
  if (!ibanModulo97Ok(iban)) {
    $("#iban").addClass("is-invalid");
    $("#errore_iban").addClass("invalid-feedback").text("IBAN formalmente corretto ma non supera il controllo Modulo 97.");
    return;
  }

  // OK
  $("#iban").removeClass("is-invalid");
  $("#errore_iban").removeClass("invalid-feedback").text("");
});

// (opzionale) normalizza anche su blur/incolla
$(document).on("blur paste", "#iban", function () {
  var iban = normalizeIban($(this).val());
  $(this).val(iban);
});

// Istituto di Credito
$(document).on("input", "#istituto-credito", function () {
  var val = ($(this).val() || "").trim();
  if (val.length === 0) {
    $("#istituto-credito").addClass("is-invalid");
    $("#errore_istituto_credito").addClass("invalid-feedback").text("Istituto di credito obbligatorio.");
  } else {
    $("#istituto-credito").removeClass("is-invalid");
    $("#errore_istituto_credito").removeClass("invalid-feedback").text("");
  }
});

// Agenzia
$(document).on("input", "#agenzia_banca", function () {
  var val = ($(this).val() || "").trim();
  if (val.length === 0) {
    $("#agenzia_banca").addClass("is-invalid");
    $("#errore_agenzia_banca").addClass("invalid-feedback").text("Agenzia obbligatoria.");
  } else {
    $("#agenzia_banca").removeClass("is-invalid");
    $("#errore_agenzia_banca").removeClass("invalid-feedback").text("");
  }
});